import { useState } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, CheckCircle, Sparkles, Hand, Scissors, Package, Flower2, Heart, Award, Clock, TrendingUp, Users, Star, Building2 } from 'lucide-react'
import { motion } from 'framer-motion'
import './BookProcedure.css'

const BookProcedure = () => {
  const navigate = useNavigate()
  const [selectedCategory, setSelectedCategory] = useState('')
  const [selectedDate, setSelectedDate] = useState('')
  const [isSubmitted, setIsSubmitted] = useState(false)

  const categories = [
    { 
      id: 'massage', 
      name: 'Массаж', 
      icon: Hand,
      color: '#D4A574',
      description: 'Расслабляющий и лечебный массаж',
      gradient: 'linear-gradient(135deg, #D4A574 0%, #C9A961 100%)',
      duration: '60-90 минут',
      price: 'от 3500₽'
    },
    { 
      id: 'peeling', 
      name: 'Пилинг', 
      icon: Scissors,
      color: '#C9A961',
      description: 'Глубокое очищение и обновление кожи',
      gradient: 'linear-gradient(135deg, #C9A961 0%, #D4A574 100%)',
      duration: '45-60 минут',
      price: 'от 2800₽'
    },
    { 
      id: 'wrap', 
      name: 'Обёртывание', 
      icon: Package,
      color: '#E8C5A0',
      description: 'Антицеллюлитное и моделирующее',
      gradient: 'linear-gradient(135deg, #E8C5A0 0%, #D4A574 100%)',
      duration: '75-90 минут',
      price: 'от 4200₽'
    },
    { 
      id: 'aromatherapy', 
      name: 'Ароматерапия', 
      icon: Flower2,
      color: '#B8956A',
      description: 'Релаксация и восстановление энергии',
      gradient: 'linear-gradient(135deg, #B8956A 0%, #D4A574 100%)',
      duration: '50-70 минут',
      price: 'от 3200₽'
    }
  ]

  const salons = [
    { name: 'SpaLux Premium', location: 'Центр', rating: 5, specialties: 'Все процедуры', capacity: 'VIP-залы' },
    { name: 'SpaLux Center', location: 'Тверская', rating: 5, specialties: 'Массаж, Пилинг', capacity: '8 кабинетов' },
    { name: 'SpaLux Relax', location: 'Арбат', rating: 5, specialties: 'Обёртывание, Ароматерапия', capacity: '6 кабинетов' },
    { name: 'SpaLux Wellness', location: 'Сокольники', rating: 5, specialties: 'Все процедуры', capacity: '10 кабинетов' }
  ]

  const advantages = [
    { icon: Heart, title: 'Натуральные компоненты', text: 'Только органические и природные средства' },
    { icon: Award, title: 'Опытные мастера', text: 'Сертифицированные специалисты с опытом' },
    { icon: Clock, title: 'Гибкий график', text: 'Запись в удобное для вас время' },
    { icon: TrendingUp, title: 'Высокий рейтинг', text: '99% довольных клиентов' }
  ]

  const handleSubmit = (e) => {
    e.preventDefault()
    if (selectedCategory && selectedDate) {
      const savedVisits = localStorage.getItem('spaVisits')
      const visits = savedVisits ? JSON.parse(savedVisits) : []
      
      const categoryData = categories.find(c => c.id === selectedCategory)
      const randomSalon = salons[Math.floor(Math.random() * salons.length)]
      
      const newVisit = {
        id: Date.now(),
        procedure: categoryData.name,
        salon: randomSalon.name,
        date: selectedDate,
        rating: 0
      }

      visits.push(newVisit)
      localStorage.setItem('spaVisits', JSON.stringify(visits))
      
      setIsSubmitted(true)
      setTimeout(() => {
        setIsSubmitted(false)
        setSelectedCategory('')
        setSelectedDate('')
        navigate('/')
      }, 2000)
    }
  }

  const minDate = new Date().toISOString().split('T')[0]
  const maxDate = new Date(Date.now() + 90 * 24 * 60 * 60 * 1000).toISOString().split('T')[0]

  return (
    <div className="book-procedure-page">
      <div className="container">
        <motion.section 
          className="book-hero"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="book-icon-wrapper"
            initial={{ scale: 0, rotate: 180 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
          >
            <Sparkles size={88} strokeWidth={2.5} className="book-hero-icon" />
          </motion.div>
          <h1 className="book-title">Записаться на процедуру</h1>
          <p className="book-subtitle">
            Выберите категорию спа-процедуры и удобную дату для незабываемого релакса
          </p>
        </motion.section>

        <motion.section 
          className="booking-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <div className="booking-form-wrapper">
            <form className="booking-form" onSubmit={handleSubmit}>
              <div className="form-group">
                <label className="form-label">
                  <Sparkles size={28} strokeWidth={2.5} className="label-icon" />
                  Выберите категорию процедуры
                </label>
                <div className="categories-grid">
                  {categories.map((category) => {
                    const IconComponent = category.icon
                    return (
                      <motion.button
                        key={category.id}
                        type="button"
                        className={`category-card ${selectedCategory === category.id ? 'selected' : ''}`}
                        onClick={() => setSelectedCategory(category.id)}
                        whileHover={{ scale: 1.1, rotate: 4 }}
                        whileTap={{ scale: 0.98 }}
                        style={{ 
                          borderColor: selectedCategory === category.id ? category.color : 'rgba(212, 165, 116, 0.3)',
                          background: selectedCategory === category.id ? (category.color + '20') : 'rgba(255, 255, 255, 0.98)'
                        }}
                      >
                        <div className="category-icon-wrapper" style={{ background: category.color + '30' }}>
                          <IconComponent size={48} strokeWidth={2.5} color={category.color} />
                        </div>
                        <div className="category-content">
                          <h3 className="category-name">{category.name}</h3>
                          <p className="category-description">{category.description}</p>
                          <div className="category-meta">
                            <span className="category-duration">{category.duration}</span>
                            <span className="category-price">{category.price}</span>
                          </div>
                        </div>
                        {selectedCategory === category.id && (
                          <motion.div 
                            className="category-check"
                            initial={{ scale: 0 }}
                            animate={{ scale: 1 }}
                            style={{ background: category.gradient }}
                          >
                            <CheckCircle size={30} strokeWidth={2.5} color="white" />
                          </motion.div>
                        )}
                      </motion.button>
                    )
                  })}
                </div>
              </div>

              <div className="form-group">
                <label htmlFor="date" className="form-label">
                  <Calendar size={28} strokeWidth={2.5} className="label-icon" />
                  Дата процедуры
                </label>
                <input
                  type="date"
                  id="date"
                  className="form-input"
                  value={selectedDate}
                  onChange={(e) => setSelectedDate(e.target.value)}
                  min={minDate}
                  max={maxDate}
                  required
                />
              </div>

              <motion.button 
                type="submit" 
                className="submit-button" 
                disabled={!selectedCategory || !selectedDate}
                whileHover={{ scale: 1.03, y: -4 }}
                whileTap={{ scale: 0.98 }}
              >
                <span>Записаться на процедуру</span>
                <CheckCircle size={28} strokeWidth={2.5} className="btn-icon" />
              </motion.button>

              {isSubmitted && (
                <motion.div 
                  className="success-message"
                  initial={{ opacity: 0, scale: 0.9 }}
                  animate={{ opacity: 1, scale: 1 }}
                  transition={{ duration: 0.3 }}
                >
                  <CheckCircle size={36} strokeWidth={2.5} className="success-icon" />
                  <span>Вы успешно записались на процедуру!</span>
                </motion.div>
              )}
            </form>
          </div>
        </motion.section>

        <motion.section 
          className="advantages-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <h2 className="section-title">Наши преимущества</h2>
          <div className="advantages-grid">
            {advantages.map((advantage, index) => {
              const IconComponent = advantage.icon
              return (
                <motion.div
                  key={advantage.title}
                  className="advantage-card"
                  initial={{ opacity: 0, y: 20 }}
                  animate={{ opacity: 1, y: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.5 }}
                  whileHover={{ translateY: -10, boxShadow: '0 16px 40px rgba(212, 165, 116, 0.3)' }}
                >
                  <div className="advantage-icon">
                    <IconComponent size={40} strokeWidth={2.5} color="#D4A574" />
                  </div>
                  <h3 className="advantage-title">{advantage.title}</h3>
                  <p className="advantage-text">{advantage.text}</p>
                </motion.div>
              )
            })}
          </div>
        </motion.section>

        <motion.section 
          className="salons-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.5 }}
        >
          <h2 className="section-title">Наши салоны</h2>
          <div className="salons-grid">
            {salons.map((salon, index) => (
              <motion.div
                key={salon.name}
                className="salon-card"
                initial={{ opacity: 0, scale: 0.9 }}
                animate={{ opacity: 1, scale: 1 }}
                transition={{ delay: index * 0.1, duration: 0.5 }}
                whileHover={{ scale: 1.05, translateY: -8 }}
              >
                <div className="salon-avatar">
                  <Building2 size={40} strokeWidth={2.5} />
                </div>
                <h3 className="salon-name">{salon.name}</h3>
                <p className="salon-location">{salon.location}</p>
                <div className="salon-rating">
                  {Array.from({ length: 5 }, (_, i) => (
                    <Star key={i} size={16} fill="#F59E0B" stroke="#F59E0B" strokeWidth={2} />
                  ))}
                </div>
                <p className="salon-specialties">{salon.specialties}</p>
                <div className="salon-capacity">
                  <Users size={16} strokeWidth={2} />
                  <span>{salon.capacity}</span>
                </div>
              </motion.div>
            ))}
          </div>
        </motion.section>
      </div>
    </div>
  )
}

export default BookProcedure

